"use client"
import { useEffect, useState } from "react"
import type React from "react"

import { useRouter } from "next/navigation"
import { SiteHeader } from "@/components/site-header"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { PlusCircle, Trash2, Edit } from "lucide-react"

interface User {
  id: string
  email: string
  username: string
  role: string
  verified: boolean
}

interface PortfolioItem {
  id: string
  userId: string
  title: string
  description: string
  imageUrl: string
  createdAt: string
}

export default function DashboardPage() {
  const router = useRouter()
  const [user, setUser] = useState<User | null>(null)
  const [portfolioItems, setPortfolioItems] = useState<PortfolioItem[]>([])
  const [showForm, setShowForm] = useState(false)
  const [editingId, setEditingId] = useState<string | null>(null)

  const [formData, setFormData] = useState({
    title: "",
    description: "",
    imageUrl: "",
  })

  useEffect(() => {
    const currentUser = localStorage.getItem("currentUser")
    if (!currentUser) {
      router.push("/login")
      return
    }

    const userData = JSON.parse(currentUser)
    setUser(userData)

    // Load portfolio items
    const items = JSON.parse(localStorage.getItem("portfolioItems") || "[]")
    setPortfolioItems(items.filter((item: PortfolioItem) => item.userId === userData.id))
  }, [router])

  const handleLogout = () => {
    localStorage.removeItem("currentUser")
    router.push("/")
  }

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault()

    if (!user?.verified) {
      alert("You must be verified to post portfolio items. Please contact an admin.")
      return
    }

    const allItems = JSON.parse(localStorage.getItem("portfolioItems") || "[]")

    if (editingId) {
      // Update existing item
      const updatedItems = allItems.map((item: PortfolioItem) =>
        item.id === editingId ? { ...item, ...formData } : item,
      )
      localStorage.setItem("portfolioItems", JSON.stringify(updatedItems))
      setPortfolioItems(updatedItems.filter((item: PortfolioItem) => item.userId === user.id))
    } else {
      // Create new item
      const newItem: PortfolioItem = {
        id: Date.now().toString(),
        userId: user.id,
        ...formData,
        createdAt: new Date().toISOString(),
      }
      allItems.push(newItem)
      localStorage.setItem("portfolioItems", JSON.stringify(allItems))
      setPortfolioItems([...portfolioItems, newItem])
    }

    setFormData({ title: "", description: "", imageUrl: "" })
    setShowForm(false)
    setEditingId(null)
  }

  const handleEdit = (item: PortfolioItem) => {
    setFormData({
      title: item.title,
      description: item.description,
      imageUrl: item.imageUrl,
    })
    setEditingId(item.id)
    setShowForm(true)
  }

  const handleDelete = (id: string) => {
    if (confirm("Are you sure you want to delete this item?")) {
      const allItems = JSON.parse(localStorage.getItem("portfolioItems") || "[]")
      const updatedItems = allItems.filter((item: PortfolioItem) => item.id !== id)
      localStorage.setItem("portfolioItems", JSON.stringify(updatedItems))
      setPortfolioItems(portfolioItems.filter((item) => item.id !== id))
    }
  }

  if (!user) return null

  return (
    <div className="min-h-screen bg-background text-foreground relative overflow-x-hidden">
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <img
          src="/la-city-bg.jpg"
          alt=""
          className="w-full h-full object-cover"
          style={{ imageRendering: "high-quality" }}
        />
        <div className="absolute inset-0 bg-black/70"></div>
      </div>

      <SiteHeader />

      <main className="container mx-auto px-6 py-20 relative">
        <div className="max-w-6xl mx-auto space-y-8">
          {/* Header */}
          <div className="flex justify-between items-center">
            <div>
              <h1 className="text-4xl font-black text-white">Developer Dashboard</h1>
              <p className="text-white mt-2">
                Welcome, {user.username} • {user.verified ? "Verified Developer" : "Awaiting Verification"}
              </p>
            </div>
            <Button onClick={handleLogout} variant="outline" className="border-white/20 text-white bg-transparent">
              Logout
            </Button>
          </div>

          {/* Verification Notice */}
          {!user.verified && (
            <div className="bg-[#d4a017] rounded-lg p-4 border-0">
              <p className="text-white">
                Your account is pending verification. Once verified by an admin, you'll be able to post portfolio items.
              </p>
            </div>
          )}

          {/* Add New Button */}
          {user.verified && !showForm && (
            <Button
              onClick={() => setShowForm(true)}
              className="bg-[#b7ff00] hover:bg-[#b7ff00]/90 text-black font-bold"
            >
              <PlusCircle className="mr-2 h-5 w-5" />
              Add Portfolio Item
            </Button>
          )}

          {/* Form */}
          {showForm && (
            <div className="bg-black/60 backdrop-blur-md border border-white/10 rounded-lg p-8">
              <h2 className="text-2xl font-bold text-white mb-6">
                {editingId ? "Edit Portfolio Item" : "New Portfolio Item"}
              </h2>
              <form onSubmit={handleSubmit} className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="title" className="text-white">
                    Title
                  </Label>
                  <Input
                    id="title"
                    value={formData.title}
                    onChange={(e) => setFormData({ ...formData, title: e.target.value })}
                    required
                    className="bg-white/5 border-white/20 text-white"
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="description" className="text-white">
                    Description
                  </Label>
                  <Textarea
                    id="description"
                    value={formData.description}
                    onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                    required
                    rows={4}
                    className="bg-white/5 border-white/20 text-white"
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="imageUrl" className="text-white">
                    Image URL
                  </Label>
                  <Input
                    id="imageUrl"
                    type="url"
                    value={formData.imageUrl}
                    onChange={(e) => setFormData({ ...formData, imageUrl: e.target.value })}
                    required
                    className="bg-white/5 border-white/20 text-white"
                  />
                </div>

                <div className="flex gap-4">
                  <Button type="submit" className="bg-[#b7ff00] hover:bg-[#b7ff00]/90 text-black font-bold">
                    {editingId ? "Update" : "Create"}
                  </Button>
                  <Button
                    type="button"
                    onClick={() => {
                      setShowForm(false)
                      setEditingId(null)
                      setFormData({ title: "", description: "", imageUrl: "" })
                    }}
                    variant="outline"
                    className="border-white/20 text-white"
                  >
                    Cancel
                  </Button>
                </div>
              </form>
            </div>
          )}

          {/* Portfolio Items Grid */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {portfolioItems.map((item) => (
              <div
                key={item.id}
                className="bg-black/60 backdrop-blur-md border border-white/10 rounded-lg overflow-hidden"
              >
                <img src={item.imageUrl || "/placeholder.svg"} alt={item.title} className="w-full h-48 object-cover" />
                <div className="p-4 space-y-3">
                  <h3 className="text-xl font-bold text-white">{item.title}</h3>
                  <p className="text-gray-400 text-sm line-clamp-3">{item.description}</p>
                  <div className="flex gap-2">
                    <Button
                      size="sm"
                      onClick={() => handleEdit(item)}
                      variant="outline"
                      className="border-white/20 text-white"
                    >
                      <Edit className="h-4 w-4 mr-1" />
                      Edit
                    </Button>
                    <Button
                      size="sm"
                      onClick={() => handleDelete(item.id)}
                      variant="outline"
                      className="border-red-500/20 text-red-400 hover:bg-red-500/10"
                    >
                      <Trash2 className="h-4 w-4 mr-1" />
                      Delete
                    </Button>
                  </div>
                </div>
              </div>
            ))}
          </div>

          {portfolioItems.length === 0 && !showForm && (
            <div className="text-center py-12 text-gray-400">
              No portfolio items yet. {user.verified ? "Click the button above to add one!" : ""}
            </div>
          )}
        </div>
      </main>
    </div>
  )
}
