"use client"
import { useEffect, useState } from "react"
import { useRouter } from "next/navigation"
import { SiteHeader } from "@/components/site-header"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { CheckCircle, XCircle, Trash2, Search, Shield, Settings } from "lucide-react"

interface AdminUser {
  id: string
  email: string
  username: string
  role: string
  verified: boolean
  createdAt: string
  permissions?: {
    canPost: boolean
    canEdit: boolean
    canDelete: boolean
  }
}

export default function AdminPage() {
  const router = useRouter()
  const [currentUser, setCurrentUser] = useState<AdminUser | null>(null)
  const [developers, setDevelopers] = useState<AdminUser[]>([])
  const [searchQuery, setSearchQuery] = useState("")
  const [filterRole, setFilterRole] = useState<string>("all")

  useEffect(() => {
    const user = localStorage.getItem("currentUser")
    if (!user) {
      router.push("/login")
      return
    }

    const userData = JSON.parse(user)

    if (userData.role !== "admin") {
      alert("Access denied. Admin privileges required.")
      router.push("/dashboard")
      return
    }

    setCurrentUser(userData)

    // Load all users
    const users = JSON.parse(localStorage.getItem("users") || "[]")
    setDevelopers(users)
  }, [router])

  const handleVerifyToggle = (userId: string) => {
    const users = JSON.parse(localStorage.getItem("users") || "[]")
    const updatedUsers = users.map((user: AdminUser) =>
      user.id === userId ? { ...user, verified: !user.verified } : user,
    )
    localStorage.setItem("users", JSON.stringify(updatedUsers))
    setDevelopers(updatedUsers)
  }

  const handleRoleChange = (userId: string, newRole: string) => {
    if (userId === currentUser?.id && newRole !== "admin") {
      alert("You cannot remove your own admin privileges!")
      return
    }

    const users = JSON.parse(localStorage.getItem("users") || "[]")
    const updatedUsers = users.map((user: AdminUser) => (user.id === userId ? { ...user, role: newRole } : user))
    localStorage.setItem("users", JSON.stringify(updatedUsers))
    setDevelopers(updatedUsers)
  }

  const handlePermissionToggle = (userId: string, permission: keyof AdminUser["permissions"]) => {
    const users = JSON.parse(localStorage.getItem("users") || "[]")
    const updatedUsers = users.map((user: AdminUser) => {
      if (user.id === userId) {
        const permissions = user.permissions || { canPost: true, canEdit: true, canDelete: true }
        return {
          ...user,
          permissions: {
            ...permissions,
            [permission]: !permissions[permission],
          },
        }
      }
      return user
    })
    localStorage.setItem("users", JSON.stringify(updatedUsers))
    setDevelopers(updatedUsers)
  }

  const handleDeleteUser = (userId: string) => {
    if (userId === currentUser?.id) {
      alert("You cannot delete your own account!")
      return
    }

    if (confirm("Are you sure you want to delete this user? This action cannot be undone.")) {
      const users = JSON.parse(localStorage.getItem("users") || "[]")
      const updatedUsers = users.filter((user: AdminUser) => user.id !== userId)
      localStorage.setItem("users", JSON.stringify(updatedUsers))
      setDevelopers(updatedUsers)

      // Also delete their portfolio items
      const portfolioItems = JSON.parse(localStorage.getItem("portfolioItems") || "[]")
      const updatedItems = portfolioItems.filter((item: any) => item.userId !== userId)
      localStorage.setItem("portfolioItems", JSON.stringify(updatedItems))
    }
  }

  const filteredDevelopers = developers.filter((dev) => {
    const matchesSearch =
      dev.username.toLowerCase().includes(searchQuery.toLowerCase()) ||
      dev.email.toLowerCase().includes(searchQuery.toLowerCase())
    const matchesRole = filterRole === "all" || dev.role === filterRole
    return matchesSearch && matchesRole
  })

  const stats = {
    total: developers.length,
    verified: developers.filter((d) => d.verified).length,
    unverified: developers.filter((d) => !d.verified).length,
    admins: developers.filter((d) => d.role === "admin").length,
  }

  if (!currentUser) return null

  return (
    <div className="min-h-screen bg-background text-foreground relative overflow-x-hidden">
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <img
          src="/la-city-bg.jpg"
          alt=""
          className="w-full h-full object-cover"
          style={{ imageRendering: "high-quality" }}
        />
        <div className="absolute inset-0 bg-black/70"></div>
      </div>

      <SiteHeader />

      <main className="container mx-auto px-6 py-20 relative">
        <div className="max-w-7xl mx-auto space-y-8">
          <div className="flex justify-between items-center">
            <div>
              <h1 className="text-4xl font-black text-white">Admin Dashboard</h1>
              <p className="text-gray-400 mt-2">Manage developers and permissions</p>
            </div>
            <Button
              onClick={() => router.push("/dashboard")}
              variant="outline"
              className="border-white/20 text-white bg-transparent"
            >
              Back to Dashboard
            </Button>
          </div>

          {/* Stats Cards */}
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div className="bg-black/60 backdrop-blur-md border border-white/10 rounded-lg p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-400 text-sm">Total Users</p>
                  <p className="text-3xl font-bold text-white mt-1">{stats.total}</p>
                </div>
                <Settings className="h-8 w-8 text-[#b7ff00]" />
              </div>
            </div>
            <div className="bg-black/60 backdrop-blur-md border border-white/10 rounded-lg p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-400 text-sm">Verified</p>
                  <p className="text-3xl font-bold text-white mt-1">{stats.verified}</p>
                </div>
                <CheckCircle className="h-8 w-8 text-green-500" />
              </div>
            </div>
            <div className="bg-black/60 backdrop-blur-md border border-white/10 rounded-lg p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-400 text-sm">Unverified</p>
                  <p className="text-3xl font-bold text-white mt-1">{stats.unverified}</p>
                </div>
                <XCircle className="h-8 w-8 text-yellow-500" />
              </div>
            </div>
            <div className="bg-black/60 backdrop-blur-md border border-white/10 rounded-lg p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-gray-400 text-sm">Admins</p>
                  <p className="text-3xl font-bold text-white mt-1">{stats.admins}</p>
                </div>
                <Shield className="h-8 w-8 text-[#b7ff00]" />
              </div>
            </div>
          </div>

          {/* Filters */}
          <div className="bg-black/60 backdrop-blur-md border border-white/10 rounded-lg p-6">
            <div className="flex flex-col md:flex-row gap-4">
              <div className="flex-1">
                <Label htmlFor="search" className="text-white mb-2 block">
                  Search Users
                </Label>
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
                  <Input
                    id="search"
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    placeholder="Search by name or email..."
                    className="bg-white/5 border-white/20 text-white pl-10"
                  />
                </div>
              </div>
              <div>
                <Label htmlFor="role-filter" className="text-white mb-2 block">
                  Filter by Role
                </Label>
                <select
                  id="role-filter"
                  value={filterRole}
                  onChange={(e) => setFilterRole(e.target.value)}
                  className="w-full md:w-48 px-3 py-2 bg-white/5 border border-white/20 text-white rounded-md"
                >
                  <option value="all">All Roles</option>
                  <option value="developer">Developer</option>
                  <option value="admin">Admin</option>
                  <option value="moderator">Moderator</option>
                </select>
              </div>
            </div>
          </div>

          {/* Users Table */}
          <div className="bg-black/60 backdrop-blur-md border border-white/10 rounded-lg overflow-hidden">
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-white/5 border-b border-white/10">
                  <tr>
                    <th className="text-left p-4 text-white font-semibold">User</th>
                    <th className="text-left p-4 text-white font-semibold">Status</th>
                    <th className="text-left p-4 text-white font-semibold">Role</th>
                    <th className="text-left p-4 text-white font-semibold">Permissions</th>
                    <th className="text-left p-4 text-white font-semibold">Joined</th>
                    <th className="text-left p-4 text-white font-semibold">Actions</th>
                  </tr>
                </thead>
                <tbody>
                  {filteredDevelopers.map((dev) => (
                    <tr key={dev.id} className="border-b border-white/10 hover:bg-white/5">
                      <td className="p-4">
                        <div>
                          <p className="text-white font-medium">{dev.username}</p>
                          <p className="text-gray-400 text-sm">{dev.email}</p>
                        </div>
                      </td>
                      <td className="p-4">
                        <button
                          onClick={() => handleVerifyToggle(dev.id)}
                          className={`px-3 py-1 rounded-full text-xs font-medium flex items-center gap-1 ${
                            dev.verified
                              ? "bg-green-500/20 text-green-400 border border-green-500/30"
                              : "bg-yellow-500/20 text-yellow-400 border border-yellow-500/30"
                          }`}
                        >
                          {dev.verified ? <CheckCircle className="h-3 w-3" /> : <XCircle className="h-3 w-3" />}
                          {dev.verified ? "Verified" : "Unverified"}
                        </button>
                      </td>
                      <td className="p-4">
                        <select
                          value={dev.role}
                          onChange={(e) => handleRoleChange(dev.id, e.target.value)}
                          className="px-3 py-1 bg-white/10 border border-white/20 text-white rounded text-sm"
                          disabled={dev.id === currentUser.id}
                        >
                          <option value="developer">Developer</option>
                          <option value="moderator">Moderator</option>
                          <option value="admin">Admin</option>
                        </select>
                      </td>
                      <td className="p-4">
                        <div className="flex gap-2">
                          <button
                            onClick={() => handlePermissionToggle(dev.id, "canPost")}
                            className={`px-2 py-1 rounded text-xs ${
                              dev.permissions?.canPost !== false
                                ? "bg-[#b7ff00]/20 text-[#b7ff00] border border-[#b7ff00]/30"
                                : "bg-red-500/20 text-red-400 border border-red-500/30"
                            }`}
                            title="Can Post"
                          >
                            Post
                          </button>
                          <button
                            onClick={() => handlePermissionToggle(dev.id, "canEdit")}
                            className={`px-2 py-1 rounded text-xs ${
                              dev.permissions?.canEdit !== false
                                ? "bg-[#b7ff00]/20 text-[#b7ff00] border border-[#b7ff00]/30"
                                : "bg-red-500/20 text-red-400 border border-red-500/30"
                            }`}
                            title="Can Edit"
                          >
                            Edit
                          </button>
                          <button
                            onClick={() => handlePermissionToggle(dev.id, "canDelete")}
                            className={`px-2 py-1 rounded text-xs ${
                              dev.permissions?.canDelete !== false
                                ? "bg-[#b7ff00]/20 text-[#b7ff00] border border-[#b7ff00]/30"
                                : "bg-red-500/20 text-red-400 border border-red-500/30"
                            }`}
                            title="Can Delete"
                          >
                            Delete
                          </button>
                        </div>
                      </td>
                      <td className="p-4">
                        <p className="text-gray-400 text-sm">{new Date(dev.createdAt).toLocaleDateString()}</p>
                      </td>
                      <td className="p-4">
                        <Button
                          size="sm"
                          onClick={() => handleDeleteUser(dev.id)}
                          disabled={dev.id === currentUser.id}
                          variant="outline"
                          className="border-red-500/20 text-red-400 hover:bg-red-500/10"
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>

          {filteredDevelopers.length === 0 && (
            <div className="text-center py-12 text-gray-400">No users found matching your search.</div>
          )}
        </div>
      </main>
    </div>
  )
}
